/*
 * Decompiled with CFR 0.152.
 */
package org.opennms.karaf.extender;

import com.google.common.collect.Lists;
import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.nio.file.DirectoryStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Dictionary;
import java.util.EnumSet;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import org.apache.karaf.features.FeaturesService;
import org.opennms.karaf.extender.Feature;
import org.opennms.karaf.extender.Repository;
import org.ops4j.pax.url.mvn.MavenResolver;
import org.osgi.service.cm.Configuration;
import org.osgi.service.cm.ConfigurationAdmin;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class KarafExtender {
    private static final Logger LOG = LoggerFactory.getLogger(KarafExtender.class);
    private static final String PAX_MVN_PID = "org.ops4j.pax.url.mvn";
    private static final String PAX_MVN_REPOSITORIES = "org.ops4j.pax.url.mvn.repositories";
    public static final String FEATURES_URI = "features.uri";
    public static final String FEATURES_BOOT = "features.boot";
    private static final String COMMENT_REGEX = "^\\s*(#.*)?$";
    private static final Pattern FEATURE_VERSION_PATTERN = Pattern.compile("(.*?)(/(.*))?");
    private final Path m_karafHome = Paths.get(System.getProperty("karaf.home"), new String[0]);
    private final Path m_repositories = this.m_karafHome.resolve("repositories");
    private final Path m_featuresBootDotD = this.m_karafHome.resolve(Paths.get("etc", "featuresBoot.d"));
    private ConfigurationAdmin m_configurationAdmin;
    private MavenResolver m_mavenResolver;
    private FeaturesService m_featuresService;

    public void init() throws InterruptedException {
        List<Repository> repositories;
        Objects.requireNonNull(this.m_configurationAdmin, "configurationAdmin");
        Objects.requireNonNull(this.m_mavenResolver, "mavenResolver");
        Objects.requireNonNull(this.m_featuresService, "featuresService");
        try {
            repositories = this.getRepositories();
        }
        catch (IOException e) {
            LOG.error("Failed to retrieve the list of repositories. Aborting.", (Throwable)e);
            return;
        }
        List<Feature> featuresBoot = repositories.stream().flatMap(r -> r.getFeaturesBoot().stream()).collect(Collectors.toList());
        try {
            featuresBoot.addAll(this.getFeaturesBoot());
        }
        catch (IOException e) {
            LOG.error("Failed to retrieve the list of features to boot. Aborting.", (Throwable)e);
            return;
        }
        this.filterFeatures(featuresBoot);
        StringBuilder mavenReposSb = new StringBuilder();
        for (Repository repository : repositories) {
            if (mavenReposSb.length() != 0) {
                mavenReposSb.append(",");
            }
            mavenReposSb.append(repository.toMavenUri());
        }
        LOG.info("Updating Maven repositories to include: {}", (Object)mavenReposSb);
        try {
            Configuration config = this.m_configurationAdmin.getConfiguration(PAX_MVN_PID);
            if (config == null) {
                throw new IOException("The OSGi configuration (admin) registry was found for pid org.ops4j.pax.url.mvn, but a configuration could not be located/generated.  This shouldn't happen.");
            }
            Dictionary props = config.getProperties();
            props.put(PAX_MVN_REPOSITORIES, mavenReposSb.toString());
            config.update(props);
        }
        catch (IOException e) {
            LOG.error("Failed to update the list of Maven repositories to '{}'. Aborting.", (Object)mavenReposSb, (Object)e);
            return;
        }
        LOG.info("Waiting up-to 30 seconds for the Maven repositories to be updated...");
        Thread.sleep(2000L);
        for (int i = 28; i > 0 && !this.canResolveAllFeatureUris(repositories); --i) {
            Thread.sleep(1000L);
        }
        for (Repository repository : repositories) {
            for (URI featureUri : repository.getFeatureUris()) {
                try {
                    LOG.info("Adding feature repository: {}", (Object)featureUri);
                    this.m_featuresService.addRepository(featureUri);
                    this.m_featuresService.refreshRepository(featureUri);
                }
                catch (Exception e) {
                    LOG.error("Failed to add feature repository '{}'. Skipping.", (Object)featureUri, (Object)e);
                }
            }
        }
        final Set featuresToInstall = featuresBoot.stream().map(f -> f.getVersion() != null ? f.getName() + "/" + f.getVersion() : f.getName()).collect(Collectors.toCollection(LinkedHashSet::new));
        CompletableFuture.runAsync(new Runnable(){

            @Override
            public void run() {
                try {
                    LOG.info("Installing features: {}", (Object)featuresToInstall);
                    KarafExtender.this.m_featuresService.installFeatures(featuresToInstall, EnumSet.noneOf(FeaturesService.Option.class));
                }
                catch (Exception e) {
                    LOG.error("Failed to install one or more features.", (Throwable)e);
                }
            }
        });
    }

    private boolean canResolveAllFeatureUris(List<Repository> repositories) {
        for (Repository repository : repositories) {
            for (URI featureUri : repository.getFeatureUris()) {
                try {
                    if (this.m_mavenResolver.resolve(featureUri.toString()) != null) continue;
                    return false;
                }
                catch (Exception e) {
                    return false;
                }
            }
        }
        return true;
    }

    public List<Repository> getRepositories() throws IOException {
        List<Path> repositoryPaths = KarafExtender.getFoldersIn(this.m_repositories);
        LinkedList repositories = Lists.newLinkedList();
        for (Path repositoryPath : repositoryPaths) {
            try {
                Path featuresBootPath;
                LinkedList featureUris = Lists.newLinkedList();
                Path featuresUriPath = repositoryPath.resolve(FEATURES_URI);
                if (featuresUriPath.toFile().isFile()) {
                    for (String line : KarafExtender.getLinesIn(featuresUriPath)) {
                        featureUris.add(new URI(line));
                    }
                }
                List<Feature> featuresBoot = (featuresBootPath = repositoryPath.resolve(FEATURES_BOOT)).toFile().isFile() ? this.getFeaturesIn(featuresBootPath) : Collections.emptyList();
                repositories.add(new Repository(repositoryPath, featureUris, featuresBoot));
            }
            catch (URISyntaxException e) {
                LOG.error("Failed to generate one or more feature URIs for repository {}. Skipping.", (Object)repositoryPath, (Object)e);
            }
        }
        return repositories;
    }

    public List<Feature> getFeaturesIn(Path featuresBootFile) throws IOException {
        LinkedList features = Lists.newLinkedList();
        for (String line : KarafExtender.getLinesIn(featuresBootFile)) {
            Matcher m = FEATURE_VERSION_PATTERN.matcher(line);
            if (!m.matches()) continue;
            if (m.group(3) == null) {
                features.add(new Feature(m.group(1)));
                continue;
            }
            features.add(new Feature(m.group(1), m.group(3)));
        }
        return features;
    }

    public List<Feature> getFeaturesBoot() throws IOException {
        LinkedList features = Lists.newLinkedList();
        for (Path featuresBootFile : KarafExtender.getFilesIn(this.m_featuresBootDotD)) {
            features.addAll(this.getFeaturesIn(featuresBootFile));
        }
        return features;
    }

    public void filterFeatures(List<Feature> features) {
        HashSet<Feature> featuresToExclude = new HashSet<Feature>();
        Iterator<Feature> it = features.iterator();
        while (it.hasNext()) {
            Feature feature = it.next();
            if (!feature.getName().startsWith("!") || feature.getName().length() <= 1) continue;
            featuresToExclude.add(new Feature(feature.getName().substring(1), feature.getVersion()));
            it.remove();
        }
        features.removeAll(featuresToExclude);
    }

    private static List<Path> getFilesIn(Path folder) throws IOException {
        ArrayList<Path> files = new ArrayList<Path>();
        try (DirectoryStream<Path> directoryStream = Files.newDirectoryStream(folder);){
            for (Path path : directoryStream) {
                if (path.toFile().isDirectory() || path.getFileName().toString().startsWith(".")) continue;
                files.add(path);
            }
        }
        Collections.sort(files);
        return files;
    }

    private static List<Path> getFoldersIn(Path folder) throws IOException {
        LinkedList paths = Lists.newLinkedList();
        try (DirectoryStream<Path> directoryStream = Files.newDirectoryStream(folder);){
            for (Path path : directoryStream) {
                if (!path.toFile().isDirectory() || path.getFileName().toString().startsWith(".")) continue;
                paths.add(path);
            }
        }
        Collections.sort(paths);
        return paths;
    }

    private static List<String> getLinesIn(Path file) throws IOException {
        return Files.readAllLines(file).stream().filter(l -> !l.matches(COMMENT_REGEX)).collect(Collectors.toList());
    }

    public void setConfigurationAdmin(ConfigurationAdmin configurationAdmin) {
        this.m_configurationAdmin = configurationAdmin;
    }

    public void setMavenResolver(MavenResolver mavenResolver) {
        this.m_mavenResolver = mavenResolver;
    }

    public void setFeaturesService(FeaturesService featuresService) {
        this.m_featuresService = featuresService;
    }
}

